<?php
declare(strict_types=1);
session_start();

require_once dirname(__DIR__) . '/app/db.php';
require_once dirname(__DIR__) . '/app/helpers.php';
require_once dirname(__DIR__) . '/app/lib/generator.php';

$pdo = db();
$r = $_GET['r'] ?? '';

function render(string $view, array $vars=[]): void {
  extract($vars);
  require dirname(__DIR__) . '/app/views/' . $view . '.php';
}

try {
  switch ($r) {
    case '':
      render('home');
      break;

    case 'debtors':
      $debtors = $pdo->query("SELECT * FROM debtors ORDER BY id_number")->fetchAll();
      render('debtors', compact('debtors'));
      break;

    case 'debtor_new':
      $mode = 'new';
      $debtor = [];
      render('debtor_form', compact('mode','debtor'));
      break;

    case 'debtor_edit':
      $id = (int)($_GET['id'] ?? 0);
      $st = $pdo->prepare("SELECT * FROM debtors WHERE id=:id");
      $st->execute([':id'=>$id]);
      $debtor = $st->fetch();
      if (!$debtor) { flash_set('err','Deudor no encontrado.'); redirect('index.php?r=debtors'); }
      $mode = 'edit';
      render('debtor_form', compact('mode','debtor'));
      break;

    case 'debtor_save':
      if ($_SERVER['REQUEST_METHOD'] !== 'POST') redirect('index.php?r=debtors');

      $id = isset($_POST['id']) && $_POST['id'] !== '' ? (int)$_POST['id'] : null;
      $id_type = (int)($_POST['id_type'] ?? 11);
      $id_number = trim((string)($_POST['id_number'] ?? ''));
      $name = trim((string)($_POST['name'] ?? ''));
      $situation_code = (string)($_POST['situation_code'] ?? '01');
      $art26_flag = (int)($_POST['art26_flag'] ?? 0);
      $recat_flag = (int)($_POST['recat_flag'] ?? 0);
      $days_overdue = ($_POST['days_overdue'] ?? '') === '' ? null : (int)$_POST['days_overdue'];
      $sit_no_re = (string)($_POST['situation_without_reclass'] ?? '00');

      if ($id_number === '' || $name === '') throw new RuntimeException('Faltan datos obligatorios.');
      if (strlen($id_number) > 11) throw new RuntimeException('Número de identificación: máximo 11 caracteres.');
      if (!in_array($situation_code, ['01','21','03','04','05'], true)) throw new RuntimeException('Situación inválida.');

      if ($recat_flag === 0) {
        $days_overdue = null;
        $sit_no_re = '00';
      } else {
        if ($days_overdue === null) throw new RuntimeException('Si Recategorización=1, debés informar días de atraso.');
        if (!in_array($sit_no_re, ['01','21','03','04','05'], true)) throw new RuntimeException('Situación sin reclasificar inválida.');
      }

      if ($id === null) {
        $st = $pdo->prepare("INSERT INTO debtors(id_type,id_number,name,situation_code,art26_flag,recat_flag,days_overdue,situation_without_reclass,updated_at)
                             VALUES(:t,:n,:nm,:s,:a,:r,:d,:snr,datetime('now'))");
        $st->execute([
          ':t'=>$id_type, ':n'=>$id_number, ':nm'=>$name, ':s'=>$situation_code, ':a'=>$art26_flag,
          ':r'=>$recat_flag, ':d'=>$days_overdue, ':snr'=>$sit_no_re
        ]);
        flash_set('ok','Deudor creado.');
      } else {
        $st = $pdo->prepare("UPDATE debtors SET id_type=:t,id_number=:n,name=:nm,situation_code=:s,art26_flag=:a,recat_flag=:r,days_overdue=:d,situation_without_reclass=:snr,updated_at=datetime('now')
                             WHERE id=:id");
        $st->execute([
          ':t'=>$id_type, ':n'=>$id_number, ':nm'=>$name, ':s'=>$situation_code, ':a'=>$art26_flag,
          ':r'=>$recat_flag, ':d'=>$days_overdue, ':snr'=>$sit_no_re, ':id'=>$id
        ]);
        flash_set('ok','Deudor actualizado.');
      }
      redirect('index.php?r=debtors');
      break;

    case 'debtor_delete':
      $id = (int)($_GET['id'] ?? 0);
      $st = $pdo->prepare("DELETE FROM debtors WHERE id=:id");
      $st->execute([':id'=>$id]);
      flash_set('ok','Deudor eliminado.');
      redirect('index.php?r=debtors');
      break;

    case 'assists':
      $period = (string)($_GET['period'] ?? date('Ym'));
      if (!is_valid_period($period)) { $period = date('Ym'); }

      $debtors = $pdo->query("SELECT id,id_number,name FROM debtors ORDER BY id_number")->fetchAll();
      $assists = $pdo->query("SELECT * FROM assists ORDER BY code")->fetchAll();

      $st = $pdo->prepare("SELECT da.id, da.debtor_id, da.assist_code, da.amount_pesos,
                                  d.id_number, d.name,
                                  a.name AS assist_name
                           FROM debtor_assists da
                           JOIN debtors d ON d.id = da.debtor_id
                           JOIN assists a ON a.code = da.assist_code
                           WHERE da.period=:p
                           ORDER BY d.id_number, da.assist_code");
      $st->execute([':p'=>$period]);
      $rows = $st->fetchAll();
      render('assists', compact('period','debtors','assists','rows'));
      break;

    case 'assist_edit':
      $period = (string)($_GET['period'] ?? date('Ym'));
      $id = (int)($_GET['id'] ?? 0);
      $st = $pdo->prepare("SELECT * FROM debtor_assists WHERE id=:id");
      $st->execute([':id'=>$id]);
      $row = $st->fetch();
      if (!$row) { flash_set('err','Registro no encontrado.'); redirect('index.php?r=assists&period='.$period); }
      $debtors = $pdo->query("SELECT id,id_number,name FROM debtors ORDER BY id_number")->fetchAll();
      $assists = $pdo->query("SELECT * FROM assists ORDER BY code")->fetchAll();
      render('assist_form', compact('period','row','debtors','assists'));
      break;

    case 'assist_save':
      if ($_SERVER['REQUEST_METHOD'] !== 'POST') redirect('index.php?r=assists');

      $id = isset($_POST['id']) && $_POST['id'] !== '' ? (int)$_POST['id'] : null;
      $period = (string)($_POST['period'] ?? date('Ym'));
      $debtor_id = (int)($_POST['debtor_id'] ?? 0);
      $assist_code = (string)($_POST['assist_code'] ?? '');
      $amount_pesos = (int)($_POST['amount_pesos'] ?? 0);

      if (!is_valid_period($period)) throw new RuntimeException('Período inválido.');
      if ($debtor_id <= 0) throw new RuntimeException('Seleccioná un deudor.');
      if ($assist_code === '') throw new RuntimeException('Seleccioná asistencia.');
      if ($amount_pesos < 0) throw new RuntimeException('Importe inválido.');

      if ($id === null) {
        $st = $pdo->prepare("INSERT INTO debtor_assists(debtor_id,period,assist_code,amount_pesos)
                             VALUES(:d,:p,:a,:m)
                             ON CONFLICT(debtor_id,period,assist_code) DO UPDATE SET amount_pesos=excluded.amount_pesos");
        $st->execute([':d'=>$debtor_id, ':p'=>$period, ':a'=>$assist_code, ':m'=>$amount_pesos]);
        flash_set('ok','Importe guardado.');
      } else {
        $st = $pdo->prepare("UPDATE debtor_assists SET debtor_id=:d, period=:p, assist_code=:a, amount_pesos=:m
                             WHERE id=:id");
        $st->execute([':d'=>$debtor_id, ':p'=>$period, ':a'=>$assist_code, ':m'=>$amount_pesos, ':id'=>$id]);
        flash_set('ok','Importe actualizado.');
      }
      redirect('index.php?r=assists&period=' . $period);
      break;

    case 'assist_delete':
      $period = (string)($_GET['period'] ?? date('Ym'));
      $id = (int)($_GET['id'] ?? 0);
      $st = $pdo->prepare("DELETE FROM debtor_assists WHERE id=:id");
      $st->execute([':id'=>$id]);
      flash_set('ok','Importe eliminado.');
      redirect('index.php?r=assists&period=' . $period);
      break;

    case 'rates':
      $rates = $pdo->query("SELECT * FROM rates ORDER BY period DESC")->fetchAll();
      render('rates', compact('rates'));
      break;

    case 'rate_edit':
      $period = (string)($_GET['period'] ?? '');
      $st = $pdo->prepare("SELECT * FROM rates WHERE period=:p");
      $st->execute([':p'=>$period]);
      $rate = $st->fetch();
      if (!$rate) { flash_set('err','No existe esa tasa.'); redirect('index.php?r=rates'); }
      render('rate_form', compact('rate'));
      break;

    case 'rate_save':
      if ($_SERVER['REQUEST_METHOD'] !== 'POST') redirect('index.php?r=rates');
      $period = (string)($_POST['period'] ?? '');
      $no_real = (int)($_POST['no_real_guarantee'] ?? 0);
      $weighted = trim((string)($_POST['weighted_rate'] ?? '000,00'));

      if (!is_valid_period($period)) throw new RuntimeException('Período inválido.');
      if (!in_array($no_real, [0,1], true)) throw new RuntimeException('Campo 1 inválido.');
      if (!preg_match('/^\d{3},\d{2}$/', $weighted)) throw new RuntimeException('Formato de tasa inválido (EEE,DD).');

      $st = $pdo->prepare("INSERT INTO rates(period,no_real_guarantee,weighted_rate)
                           VALUES(:p,:n,:w)
                           ON CONFLICT(period) DO UPDATE SET no_real_guarantee=excluded.no_real_guarantee, weighted_rate=excluded.weighted_rate");
      $st->execute([':p'=>$period, ':n'=>$no_real, ':w'=>$weighted]);
      flash_set('ok','Tasa guardada.');
      redirect('index.php?r=rates');
      break;

    case 'rate_delete':
      $period = (string)($_GET['period'] ?? '');
      $st = $pdo->prepare("DELETE FROM rates WHERE period=:p");
      $st->execute([':p'=>$period]);
      flash_set('ok','Tasa eliminada.');
      redirect('index.php?r=rates');
      break;

    case 'export':
      render('export', ['generated'=>[]]);
      break;

    case 'export_run':
      if ($_SERVER['REQUEST_METHOD'] !== 'POST') redirect('index.php?r=export');
      $period = (string)($_POST['period'] ?? '');
      $files = export_files_for_period($period);

      $generated = [];
      foreach ($files as $f) {
        $url = 'download.php?f=' . urlencode(basename($f['path']));
        $generated[] = [
          'name' => $f['name'],
          'url' => $url,
          'size' => number_format((int)$f['bytes'], 0, ',', '.') . ' bytes'
        ];
      }
      render('export', compact('generated'));
      break;

    case 'settings':
      $settings = [];
      foreach ($pdo->query("SELECT key,value FROM settings") as $row) {
        $settings[$row['key']] = $row['value'];
      }
      $cfg = require dirname(__DIR__) . '/app/config.php';
      $min_total_pesos = get_min_total_pesos();
      render('settings', compact('settings','min_total_pesos'));
      break;

    case 'settings_save':
      if ($_SERVER['REQUEST_METHOD'] !== 'POST') redirect('index.php?r=settings');
      $entity = preg_replace('/\D/', '', (string)($_POST['entity_code'] ?? '00000'));
      $entity = str_pad(substr($entity,0,5), 5, '0', STR_PAD_LEFT);
      $min = (int)($_POST['min_total_pesos'] ?? get_min_total_pesos());

      $st = $pdo->prepare("INSERT INTO settings(key,value) VALUES('entity_code',:v)
                           ON CONFLICT(key) DO UPDATE SET value=excluded.value");
      $st->execute([':v'=>$entity]);

      $st2 = $pdo->prepare("INSERT INTO settings(key,value) VALUES('min_total_pesos',:v)
                           ON CONFLICT(key) DO UPDATE SET value=excluded.value");
      $st2->execute([':v'=>(string)$min]);

      flash_set('ok','Configuración guardada.');
      redirect('index.php?r=settings');
      break;

    default:
      http_response_code(404);
      echo "404";
  }
} catch (Throwable $e) {
  flash_set('err', $e->getMessage());
  $back = $_SERVER['HTTP_REFERER'] ?? 'index.php';
  redirect($back);
}
