<?php
declare(strict_types=1);

function h(string $s): string {
  return htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
}

function redirect(string $to): never {
  header('Location: ' . $to);
  exit;
}

function flash_set(string $type, string $msg): void {
  $_SESSION['flash'] = ['type'=>$type, 'msg'=>$msg];
}

function flash_get(): ?array {
  if (!isset($_SESSION['flash'])) return null;
  $f = $_SESSION['flash'];
  unset($_SESSION['flash']);
  return $f;
}

function normalize_name(string $s): string {
  // Mayúsculas, sin tildes, sin ; ni saltos de línea
  $s = mb_strtoupper(trim($s), 'UTF-8');
  $s = str_replace(["\r","\n",";","\t","\"","'"], " ", $s);
  // quitar tildes
  $from = ['Á','É','Í','Ó','Ú','Ü','Ñ'];
  $to   = ['A','E','I','O','U','U','N'];
  $s = str_replace($from,$to,$s);
  // colapsar espacios
  $s = preg_replace('/\s+/u',' ', $s);
  return trim($s);
}

function is_valid_period(string $p): bool {
  return (bool)preg_match('/^\d{6}$/', $p) && (int)substr($p,4,2) >=1 && (int)substr($p,4,2) <= 12;
}

function pesos_to_thousands_int(int $pesos): int {
  // Miles de pesos sin decimales. Redondeo estándar: 0.5 hacia arriba
  // Ej: 2.658.500 => 2659
  return (int)round($pesos / 1000, 0, PHP_ROUND_HALF_UP);
}

function crlf_join(array $lines): string {
  return implode("\r\n", $lines) . "\r\n";
}

function export_encode(string $utf8, string $targetEncoding): string {
  // Convertir UTF-8 a ANSI-1252 (Windows-1252)
  $out = iconv('UTF-8', $targetEncoding . '//TRANSLIT//IGNORE', $utf8);
  if ($out === false) return $utf8; // fallback
  return $out;
}
